import { mysqlPool } from '../../mysql.js';
import { ajvValid } from '../../utils.js';

export const apiSchema = {
    // 元数据
    meta: {
        tags: ['用户'],
        summary: '用户登录'
    },
    // 请求协议
    request: {
        type: 'object',
        properties: {
            username: {
                type: 'string',
                title: '用户名',
                minLength: 2,
                maxLength: 20
            },
            password: {
                type: 'string',
                title: '密码',
                minLength: 6,
                maxLength: 50
            }
        },
        required: ['username', 'password'],
        additionalProperties: false
    },
    // 返回协议
    response: {
        type: 'object',
        properties: {
            token: {
                type: 'string',
                description: '登录令牌'
            }
        }
    }
};

// 接口逻辑
export default async (req) => {
    try {
        await ajvValid(apiSchema.request, req.body);
        // ------------------------------------------------------
        // 从连接池中获取一个数据库连接
        const db = await mysqlPool.getConnection();
        // 查询数据库是否有对应的用户数据
        const [rows] = await db.query({
            sql: 'SELECT * FROM `user` WHERE `username`=:username LIMIT 1',
            values: {
                username: req.body.username
            }
        });
        // ------------------------------------------------------
        // 如果查到了用户数据，说明该用户名已注册
        if (rows.length <= 0) {
            return {
                code: 1,
                msg: '用户未注册'
            };
        }
        // ------------------------------------------------------
        // 判断密码是否匹配
        const [user] = rows; // rows是一个数组，我们这里用user变量去取数组的第一个值
        if (user.password !== req.body.password) {
            return {
                code: 1,
                msg: '密码错误'
            };
        }
        // 释放数据库连接
        db.release();
        // 返回成功信息
        return {
            code: 0,
            msg: '登录成功',
            data: user
        };
    } catch (err) {
        if (err?.from === 'ajv') {
            return {
                code: 1,
                msg: err.data
            };
        } else {
            return {
                code: 1,
                msg: '未知错误'
            };
        }
    }
};
