import { mysqlPool } from '../../mysql.js';
import { ajvValid } from '../../utils.js';

export const apiSchema = {
    // 元数据
    meta: {
        tags: ['用户'],
        summary: '用户注册'
    },
    // 请求协议
    request: {
        type: 'object',
        properties: {
            username: {
                type: 'string',
                title: '用户名',
                minLength: 2,
                maxLength: 20
            },
            password: {
                type: 'string',
                title: '密码',
                minLength: 6,
                maxLength: 50
            }
        },
        required: ['username', 'password'],
        additionalProperties: false
    }
};

export default async (req) => {
    try {
        await ajvValid(apiSchema.request, req.body);
        // ------------------------------------------------------
        // 从连接池中获取一个数据库连接
        const db = await mysqlPool.getConnection();
        // 查询数据库是否有对应的用户数据
        const [rows] = await db.query({
            sql: 'SELECT * FROM `user` WHERE `username`=:username LIMIT 1',
            values: {
                username: req.body.username
            }
        });
        // ------------------------------------------------------
        // 如果查到了用户数据，说明该用户名已注册
        if (rows.length > 0) {
            return {
                code: 1,
                msg: '用户已注册'
            };
        }
        // ------------------------------------------------------
        // 插入用户数据
        const [result] = await db.query({
            sql: 'INSERT INTO `user` (`username`,`password`,`created_at`) VALUES (:username,:password,:created_at)',
            values: {
                username: req.body.username,
                password: req.body.password,
                created_at: Date.now()
            }
        });
        // 释放数据库连接
        db.release();
        // 返回成功信息
        return {
            code: 0,
            msg: '注册成功'
        };
    } catch (err) {
        console.log('🚀 ~ err:', err);
        return {
            code: 1,
            msg: '未知错误'
        };
    }
};
